<?php
/**
 * Plugin Name: WP Swift Contact
 * Plugin URI: https://proleads.online/wordpress-plugins
 * Description: Leichtgewichtiges, aber vollwertiges Kontaktformular mit Google reCAPTCHA v3 - Performance-optimiert und mehrsprachig
 * Version: 1.0.3
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * Author: ProLeads
 * Author URI: https://proleads.online
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: wp-swift-contact
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

// Plugin-Konstanten
define('WPSC_VERSION', '1.0.3');
define('WPSC_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WPSC_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WPSC_PLUGIN_FILE', __FILE__);

/**
 * Hauptklasse
 */
class WP_Swift_Contact {
    
    private static $instance = null;
    private $has_shortcode = false;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Textdomain für Übersetzungen laden
        add_action('plugins_loaded', [$this, 'load_textdomain']);
        
        // Hooks
        add_action('init', [$this, 'register_shortcode']);
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('wp_ajax_wpsc_submit_form', [$this, 'handle_submission']);
        add_action('wp_ajax_nopriv_wpsc_submit_form', [$this, 'handle_submission']);
        
        // Admin-Hinweise
        add_action('admin_notices', [$this, 'admin_notices']);
        
        // Conditional Loading
        add_action('wp_enqueue_scripts', [$this, 'check_for_shortcode'], 999);
        add_action('wp_footer', [$this, 'maybe_load_recaptcha'], 99);
        
        // Admin-Feld-Verwaltung
        add_action('admin_post_wpsc_save_fields', [$this, 'save_fields']);
    }
    
    /**
     * Textdomain laden
     */
    public function load_textdomain() {
        load_plugin_textdomain('wp-swift-contact', false, dirname(plugin_basename(WPSC_PLUGIN_FILE)) . '/languages');
    }
    
    /**
     * Admin-Hinweise
     */
    public function admin_notices() {
        // Prüfe ob reCAPTCHA konfiguriert ist
        $site_key = get_option('wpsc_recaptcha_site_key');
        $secret = get_option('wpsc_recaptcha_secret');
        
        if (empty($site_key) || empty($secret)) {
            $screen = get_current_screen();
            if ($screen && $screen->id !== 'settings_page_wpsc-settings') {
                ?>
                <div class="notice notice-warning is-dismissible">
                    <p>
                        <strong><?php _e('WP Swift Contact:', 'wp-swift-contact'); ?></strong> 
                        <?php _e('Bitte konfiguriere deine reCAPTCHA Keys unter', 'wp-swift-contact'); ?> 
                        <a href="<?php echo admin_url('options-general.php?page=wpsc-settings'); ?>">
                            <?php _e('Einstellungen → WP Swift Contact', 'wp-swift-contact'); ?>
                        </a>
                    </p>
                </div>
                <?php
            }
        }
        
        // Prüfe ob Brevo installiert ist
        if (!class_exists('Sendinblue') && !class_exists('SendinBlue') && !function_exists('ws_smtp_mail')) {
            $screen = get_current_screen();
            if ($screen && $screen->id === 'settings_page_wpsc-settings' && !isset($_GET['brevo-dismissed'])) {
                ?>
                <div class="notice notice-info">
                    <p>
                        <strong>📧 <?php _e('Professioneller E-Mail-Versand empfohlen!', 'wp-swift-contact'); ?></strong><br>
                        <?php _e('Für maximale Zustellbarkeit empfehlen wir Brevo (ehemals Sendinblue).', 'wp-swift-contact'); ?>
                        <a href="<?php echo admin_url('options-general.php?page=wpsc-settings#brevo'); ?>" class="button button-primary" style="margin-left: 10px;">
                            <?php _e('Mehr erfahren', 'wp-swift-contact'); ?>
                        </a>
                    </p>
                </div>
                <?php
            }
        }
    }
    
    /**
     * Shortcode registrieren
     */
    public function register_shortcode() {
        add_shortcode('swiftcontact', [$this, 'render_form']);
        add_shortcode('swift_contact', [$this, 'render_form']);
        // Legacy-Support
        add_shortcode('kontaktformular', [$this, 'render_form']);
    }
    
    /**
     * Prüfen ob Shortcode auf aktueller Seite ist
     */
    public function check_for_shortcode() {
        global $post;
        
        if (is_a($post, 'WP_Post')) {
            $has_sc = has_shortcode($post->post_content, 'swiftcontact') || 
                      has_shortcode($post->post_content, 'swift_contact') ||
                      has_shortcode($post->post_content, 'kontaktformular');
                      
            if ($has_sc) {
                $this->has_shortcode = true;
                $this->enqueue_assets();
            }
        }
    }
    
    /**
     * Assets laden (nur wenn nötig)
     */
    private function enqueue_assets() {
        // Inline CSS für minimale Requests
        add_action('wp_footer', function() {
            echo '<style id="wpsc-styles">';
            include WPSC_PLUGIN_DIR . 'assets/style.css';
            echo '</style>';
        }, 1);
        
        // Inline JS für minimale Requests
        add_action('wp_footer', function() {
            $nonce = wp_create_nonce('wpsc_submit');
            $ajax_url = admin_url('admin-ajax.php');
            $site_key = esc_js(get_option('wpsc_recaptcha_site_key'));
            
            echo '<script id="wpsc-script">';
            ?>
(function() {
    'use strict';
    
    // Warte bis Formular im DOM ist
    function initForm() {
        const form = document.getElementById('wpsc-form');
        if (!form) {
            console.error('WP Swift Contact: Form not found');
            return;
        }
        
        const submitBtn = form.querySelector('.wpsc-submit');
        const formMsg = form.querySelector('.wpsc-message');
        
        if (!submitBtn || !formMsg) {
            console.error('WP Swift Contact: Required elements not found');
            return;
        }
        
        form.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            // Fehler zurücksetzen
            form.querySelectorAll('.wpsc-error').forEach(el => el.textContent = '');
            formMsg.style.display = 'none';
            
            // Button deaktivieren
            submitBtn.disabled = true;
            submitBtn.textContent = '<?php echo esc_js(__('Wird gesendet...', 'wp-swift-contact')); ?>';
            
            try {
                // Prüfe ob grecaptcha geladen ist
                if (typeof grecaptcha === 'undefined') {
                    throw new Error('reCAPTCHA not loaded');
                }
                
                // reCAPTCHA Token holen
                const token = await grecaptcha.execute('<?php echo $site_key; ?>', {
                    action: 'contact_form'
                });
                
                // FormData erstellen
                const formData = new FormData(form);
                formData.append('action', 'wpsc_submit_form');
                formData.append('recaptcha_token', token);
                formData.append('nonce', '<?php echo $nonce; ?>');
                
                // AJAX-Anfrage
                const response = await fetch('<?php echo $ajax_url; ?>', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                // Ergebnis anzeigen
                formMsg.style.display = 'block';
                formMsg.className = 'wpsc-message ' + (result.success ? 'success' : 'error');
                formMsg.textContent = result.data.message;
                
                if (result.success) {
                    form.reset();
                    formMsg.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                } else if (result.data.errors) {
                    Object.keys(result.data.errors).forEach(field => {
                        const errorEl = form.querySelector('[data-error="' + field + '"]');
                        if (errorEl) {
                            errorEl.textContent = result.data.errors[field];
                        }
                    });
                }
            } catch (error) {
                console.error('WP Swift Contact Error:', error);
                formMsg.style.display = 'block';
                formMsg.className = 'wpsc-message error';
                formMsg.textContent = '<?php echo esc_js(__('Ein Fehler ist aufgetreten. Bitte versuchen Sie es erneut.', 'wp-swift-contact')); ?>';
            } finally {
                submitBtn.disabled = false;
                submitBtn.textContent = '<?php echo esc_js(__('Nachricht senden', 'wp-swift-contact')); ?>';
            }
        });
    }
    
    // Init when DOM ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initForm);
    } else {
        initForm();
    }
})();
            <?php
            echo '</script>';
        }, 99);
    }
    
    /**
     * reCAPTCHA conditional laden
     */
    public function maybe_load_recaptcha() {
        if (!$this->has_shortcode) {
            return;
        }
        
        $site_key = get_option('wpsc_recaptcha_site_key');
        if (empty($site_key)) {
            echo '<!-- WP Swift Contact: reCAPTCHA Site Key not configured -->';
            return;
        }
        
        // Async & defer für Core Web Vitals
        // Hinweis: Für CSP muss google.com und gstatic.com erlaubt sein
        echo '<script src="https://www.google.com/recaptcha/api.js?render=' . esc_attr($site_key) . '" async defer></script>';
    }
    
    /**
     * Formular rendern
     */
    public function render_form($atts) {
        $atts = shortcode_atts(['id' => 'wpsc-form'], $atts);
        
        // Felder aus DB holen
        global $wpdb;
        $table_name = $wpdb->prefix . 'wpsc_fields';
        $fields = $wpdb->get_results(
            "SELECT * FROM $table_name WHERE is_active = 1 ORDER BY sort_order ASC",
            ARRAY_A
        );
        
        // Debug: Prüfe ob Felder vorhanden sind
        if (empty($fields)) {
            return '<div class="wpsc-error-box">' . 
                   __('Fehler: Keine Formularfelder gefunden. Bitte aktiviere mindestens ein Feld im Admin.', 'wp-swift-contact') . 
                   '</div>';
        }
        
        ob_start();
        ?>
        <form id="<?php echo esc_attr($atts['id']); ?>" class="wpsc-form" novalidate>
            <?php wp_nonce_field('wpsc_submit', 'wpsc_nonce'); ?>
            
            <?php foreach ($fields as $field): ?>
                <div class="wpsc-field">
                    <label for="wpsc-<?php echo esc_attr($field['field_name']); ?>">
                        <?php echo esc_html($field['field_label']); ?>
                        <?php if ($field['is_required']): ?>
                            <span class="wpsc-required">*</span>
                        <?php endif; ?>
                    </label>
                    
                    <?php if ($field['field_type'] === 'textarea'): ?>
                        <textarea
                            id="wpsc-<?php echo esc_attr($field['field_name']); ?>"
                            name="<?php echo esc_attr($field['field_name']); ?>"
                            rows="<?php echo esc_attr($field['field_rows']); ?>"
                            placeholder="<?php echo esc_attr($field['placeholder']); ?>"
                            <?php echo $field['is_required'] ? 'required aria-required="true"' : ''; ?>
                        ></textarea>
                    <?php else: ?>
                        <input
                            type="<?php echo esc_attr($field['field_type']); ?>"
                            id="wpsc-<?php echo esc_attr($field['field_name']); ?>"
                            name="<?php echo esc_attr($field['field_name']); ?>"
                            placeholder="<?php echo esc_attr($field['placeholder']); ?>"
                            <?php echo $field['is_required'] ? 'required aria-required="true"' : ''; ?>
                        >
                    <?php endif; ?>
                    
                    <span class="wpsc-error" data-error="<?php echo esc_attr($field['field_name']); ?>"></span>
                </div>
            <?php endforeach; ?>
            
            <div class="wpsc-field">
                <button type="submit" class="wpsc-submit"><?php _e('Nachricht senden', 'wp-swift-contact'); ?></button>
            </div>
            
            <div class="wpsc-message" style="display: none;" role="alert" aria-live="polite"></div>
            
            <div class="wpsc-recaptcha-notice">
                <?php 
                printf(
                    __('Diese Seite ist durch reCAPTCHA geschützt. Es gelten die %s und %s von Google.', 'wp-swift-contact'),
                    '<a href="https://policies.google.com/privacy" target="_blank" rel="noopener">' . __('Datenschutzbestimmungen', 'wp-swift-contact') . '</a>',
                    '<a href="https://policies.google.com/terms" target="_blank" rel="noopener">' . __('Nutzungsbedingungen', 'wp-swift-contact') . '</a>'
                );
                ?>
            </div>
        </form>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Formular-Einreichung verarbeiten
     */
    public function handle_submission() {
        // Nonce prüfen
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wpsc_submit')) {
            wp_send_json_error(['message' => __('Sicherheitsprüfung fehlgeschlagen.', 'wp-swift-contact')]);
        }
        
        // reCAPTCHA prüfen
        if (!$this->verify_recaptcha($_POST['recaptcha_token'] ?? '')) {
            wp_send_json_error(['message' => __('reCAPTCHA-Validierung fehlgeschlagen. Bitte versuchen Sie es erneut.', 'wp-swift-contact')]);
        }
        
        // Felder validieren
        global $wpdb;
        $table_name = $wpdb->prefix . 'wpsc_fields';
        $fields = $wpdb->get_results(
            "SELECT * FROM $table_name WHERE is_active = 1",
            ARRAY_A
        );
        
        $errors = [];
        $data = [];
        
        foreach ($fields as $field) {
            $field_name = $field['field_name'];
            $value = isset($_POST[$field_name]) ? sanitize_text_field($_POST[$field_name]) : '';
            
            // Pflichtfeld-Prüfung
            if ($field['is_required'] && empty($value)) {
                $errors[$field_name] = sprintf(__('%s ist erforderlich.', 'wp-swift-contact'), $field['field_label']);
                continue;
            }
            
            // E-Mail-Validierung
            if ($field['field_type'] === 'email' && !empty($value)) {
                if (!is_email($value)) {
                    $errors[$field_name] = __('Ungültige E-Mail-Adresse.', 'wp-swift-contact');
                    continue;
                }
            }
            
            $data[$field_name] = $value;
        }
        
        if (!empty($errors)) {
            wp_send_json_error([
                'message' => __('Bitte überprüfen Sie Ihre Eingaben.', 'wp-swift-contact'),
                'errors' => $errors
            ]);
        }
        
        // E-Mail versenden
        $recipient = get_option('wpsc_recipient_email', get_option('admin_email'));
        $subject = sprintf(__('Neue Kontaktanfrage: %s', 'wp-swift-contact'), ($data['subject'] ?? __('Kontaktformular', 'wp-swift-contact')));
        
        $message = $this->build_email_body($data, $fields);
        
        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            'Reply-To: ' . ($data['email'] ?? get_option('admin_email'))
        ];
        
        $sent = wp_mail($recipient, $subject, $message, $headers);
        
        if ($sent) {
            wp_send_json_success([
                'message' => __('Vielen Dank! Ihre Nachricht wurde erfolgreich gesendet.', 'wp-swift-contact')
            ]);
        } else {
            wp_send_json_error([
                'message' => __('Fehler beim Versenden. Bitte versuchen Sie es später erneut oder kontaktieren Sie uns telefonisch.', 'wp-swift-contact')
            ]);
        }
    }
    
    /**
     * reCAPTCHA verifizieren
     */
    private function verify_recaptcha($token) {
        $secret = get_option('wpsc_recaptcha_secret');
        if (empty($secret) || empty($token)) {
            return false;
        }
        
        $response = wp_remote_post('https://www.google.com/recaptcha/api/siteverify', [
            'body' => [
                'secret' => $secret,
                'response' => $token
            ],
            'timeout' => 10
        ]);
        
        if (is_wp_error($response)) {
            error_log('WP Swift Contact reCAPTCHA Error: ' . $response->get_error_message());
            return false;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response));
        return isset($body->success) && $body->success && isset($body->score) && $body->score >= 0.5;
    }
    
    /**
     * E-Mail-Body erstellen
     */
    private function build_email_body($data, $fields) {
        $html = '<html><body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333;">';
        $html .= '<h2 style="color: #2c3e50; border-bottom: 2px solid: #3498db; padding-bottom: 10px;">' . __('Neue Kontaktanfrage', 'wp-swift-contact') . '</h2>';
        
        foreach ($fields as $field) {
            $field_name = $field['field_name'];
            if (isset($data[$field_name]) && !empty($data[$field_name])) {
                $value = esc_html($data[$field_name]);
                $label = esc_html($field['field_label']);
                
                $html .= '<div style="margin-bottom: 15px;">';
                $html .= '<strong style="color: #2c3e50;">' . $label . ':</strong><br>';
                
                if ($field['field_type'] === 'textarea') {
                    $html .= '<div style="background: #f8f9fa; padding: 10px; border-left: 3px solid #3498db; margin-top: 5px;">';
                    $html .= nl2br($value);
                    $html .= '</div>';
                } else {
                    $html .= '<span style="color: #555;">' . $value . '</span>';
                }
                
                $html .= '</div>';
            }
        }
        
        $html .= '<hr style="border: none; border-top: 1px solid #ddd; margin: 20px 0;">';
        $html .= '<p style="color: #7f8c8d; font-size: 12px;">' . sprintf(__('Gesendet am: %s', 'wp-swift-contact'), current_time('d.m.Y H:i:s')) . '</p>';
        $html .= '</body></html>';
        
        return $html;
    }
    
    /**
     * Admin-Menü
     */
    public function add_admin_menu() {
        add_options_page(
            __('WP Swift Contact', 'wp-swift-contact'),
            __('WP Swift Contact', 'wp-swift-contact'),
            'manage_options',
            'wpsc-settings',
            [$this, 'render_settings_page']
        );
    }
    
    /**
     * Einstellungen registrieren
     */
    public function register_settings() {
        register_setting('wpsc_settings', 'wpsc_recaptcha_site_key', [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        register_setting('wpsc_settings', 'wpsc_recaptcha_secret', [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        register_setting('wpsc_settings', 'wpsc_recipient_email', [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_email'
        ]);
    }
    
    /**
     * Settings-Seite rendern
     */
    public function render_settings_page() {
        include WPSC_PLUGIN_DIR . 'admin/settings-page.php';
    }
    
    /**
     * Felder speichern
     */
    public function save_fields() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Keine Berechtigung', 'wp-swift-contact'));
        }
        
        check_admin_referer('wpsc_save_fields');
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'wpsc_fields';
        
        $success = false;
        $error = '';
        
        // Neues Feld
        if (isset($_POST['add_field'])) {
            $field_name = sanitize_key($_POST['field_name']);
            
            // Prüfe ob Feldname bereits existiert
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE field_name = %s",
                $field_name
            ));
            
            if ($exists > 0) {
                $error = 'exists';
            } else {
                $result = $wpdb->insert($table_name, [
                    'field_name' => $field_name,
                    'field_label' => sanitize_text_field($_POST['field_label']),
                    'field_type' => sanitize_text_field($_POST['field_type']),
                    'is_required' => isset($_POST['is_required']) ? 1 : 0,
                    'placeholder' => sanitize_text_field($_POST['placeholder'] ?? ''),
                    'field_rows' => intval($_POST['field_rows'] ?? 5),
                    'sort_order' => intval($_POST['sort_order'] ?? 999)
                ]);
                
                if ($result === false) {
                    error_log('WP Swift Contact: Insert failed - ' . $wpdb->last_error);
                    $error = 'db_error';
                } else {
                    $success = true;
                }
            }
        }
        
        // Feld bearbeiten
        if (isset($_POST['edit_field'])) {
            $wpdb->update(
                $table_name,
                [
                    'field_label' => sanitize_text_field($_POST['field_label']),
                    'field_type' => sanitize_text_field($_POST['field_type']),
                    'is_required' => isset($_POST['is_required']) ? 1 : 0,
                    'placeholder' => sanitize_text_field($_POST['placeholder'] ?? ''),
                    'field_rows' => intval($_POST['field_rows'] ?? 5),
                    'sort_order' => intval($_POST['sort_order'])
                ],
                ['id' => intval($_POST['field_id'])]
            );
            $success = true;
        }
        
        // Feld löschen
        if (isset($_POST['delete_field'])) {
            $wpdb->delete($table_name, ['id' => intval($_POST['field_id'])]);
            $success = true;
        }
        
        // Feld aktivieren/deaktivieren
        if (isset($_POST['toggle_field'])) {
            $wpdb->update(
                $table_name,
                ['is_active' => intval($_POST['is_active'])],
                ['id' => intval($_POST['field_id'])]
            );
            $success = true;
        }
        
        // Redirect mit Status
        $redirect_url = admin_url('options-general.php?page=wpsc-settings&tab=fields');
        if ($success) {
            $redirect_url .= '&updated=1';
        } elseif ($error) {
            $redirect_url .= '&error=' . $error;
        }
        
        wp_redirect($redirect_url);
        exit;
    }
}

/**
 * Plugin-Aktivierung
 */
function wpsc_activate_plugin() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'wpsc_fields';
    $charset_collate = $wpdb->get_charset_collate();
    
    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        field_name varchar(50) NOT NULL,
        field_label varchar(100) NOT NULL,
        field_type varchar(20) NOT NULL,
        is_required tinyint(1) DEFAULT 0,
        placeholder varchar(255) DEFAULT '',
        field_rows int(3) DEFAULT 5,
        sort_order int(5) DEFAULT 0,
        is_active tinyint(1) DEFAULT 1,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE KEY field_name (field_name)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
    
    // Standard-Felder einfügen (ohne Übersetzung, da Textdomain noch nicht geladen)
    $count = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
    if ($count == 0) {
        $default_fields = [
            ['name', 'Name', 'text', 1, 'Ihr Name', 5, 10],
            ['email', 'E-Mail', 'email', 1, 'ihre@email.de', 5, 20],
            ['phone', 'Telefon', 'tel', 0, '+49 123 456789', 5, 30],
            ['subject', 'Betreff', 'text', 1, 'Worum geht es?', 5, 40],
            ['message', 'Nachricht', 'textarea', 1, 'Ihre Nachricht...', 6, 50],
        ];
        
        foreach ($default_fields as $field) {
            $result = $wpdb->insert($table_name, [
                'field_name' => $field[0],
                'field_label' => $field[1],
                'field_type' => $field[2],
                'is_required' => $field[3],
                'placeholder' => $field[4],
                'field_rows' => $field[5],
                'sort_order' => $field[6]
            ]);
            
            // Debug: Log wenn Insert fehlschlägt
            if ($result === false) {
                error_log('WP Swift Contact: Failed to insert field ' . $field[0] . ' - Error: ' . $wpdb->last_error);
            }
        }
    }
}

// Activation Hook registrieren
register_activation_hook(WPSC_PLUGIN_FILE, 'wpsc_activate_plugin');

// Plugin initialisieren
add_action('plugins_loaded', function() {
    WP_Swift_Contact::get_instance();
});
